/*
 * $QNXLicenseC:
 * Copyright 2010-2014, QNX Software Systems. All Rights Reserved.
 *
 * You must obtain a written license from and pay applicable
 * license fees to QNX Software Systems before you may reproduce,
 * modify or distribute this software, or any work that includes
 * all or part of this software.   Free development licenses are
 * available for evaluation and non-commercial purposes.  For more
 * information visit http://licensing.qnx.com or email
 * licensing@qnx.com.
 *
 * This file may contain contributions from others.  Please review
 * this entire file for other proprietary rights or license notices,
 * as well as the QNX Development Suite License Guide at
 * http://licensing.qnx.com/license-guide/ for other information.
 * $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/dispatch.h>
#include <stdio.h>
#include <sys/slog.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <hw/cypressctrl.h>

#define DEBUG_LOG(msg, ...) slogf( 99, 1, "touch_power_control:%s():%d: " msg, __FUNCTION__, __LINE__ , ##__VA_ARGS__)
#define ATTACH_POINT "cypress_ctrl_channel"

#ifdef __USAGE
%C - touch driver control sample
%C options
  -l                      Last x/y coordinates
  -r                      Reset controller
  -f <firmware.cyacd>     Flash firmware (path to firmware required)
  -i                      Request Firmware/Controller info
  -a                      Request Additonal Controller info
  -I                      Request IDAC calibration
  -D                      Request IDAC calibration data
  -b                      Request Init of Baselines only (Sensing Modes required)
  -s                      Request Controller Status Information
  -W                      Run Cp and Cp Tests and write results to file
                             /tmp/cm_data.csv          /tmp/cp_data.csv
  -R <file>               Read contents of a file and display them.
  -m <mode>               Capacitive Sensing Modes (can be ORd together):
                               1 - Mutual Cap Fine
                               2 - Mutual Cap Button
                               4 - Self Cap

                          E.g. Mutual Cap Button with Self Cap
                                  -m 6

                          Used only when firmware requires a recalibration of
                          the IDACs.

                          * Must be used with firmware update, not a stand
                            alone command

                          ======================================================
                          =                                                    =
                          = Warning: Must be done in a controlled environment! =
                          =                                                    =
                          ======================================================
#endif

int
dump_data (uint8_t *data, int size)
{
	char *buf;
	int i;

	if (!size)
		return 0;

	if (128 < size) {
		fprintf (stderr, "Limiting the size to 128\n");
		size = 128;
	}

	buf = (char *) malloc (size);

	if (buf == NULL) {
		fprintf (stderr, "Failed to allocate buffer space to dump data");
		return -1;
	}

	sprintf (buf, "%x ", data[0]);

	for (i = 1; i < size; i++) {
		sprintf ((buf + strlen(buf)), "%x ", data[i]);
	}

	fprintf (stdout, "Raw Data: %s\n", buf);

	free (buf);

	return 0;
}

/* This is sort of a catch all function, that will be expanded over time */
int
cypress_request_controller_status(int coid)
{
	int reply;
	cypress_msg_u msg;
	cypress_msg_add_controller_status_t controller_status;

	msg.cmd = CYPRESS_CTRL_CONTROLLER_STATUS_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), &(controller_status), sizeof (controller_status));

	if (EOK != reply) {
		printf ("Request for controller info failed; reply=%d\n", reply);
		return -1;
	}

	fprintf (stdout, "IDAC calibration status: ");
	switch (controller_status.idac_calibration_status) {
	case -1:
		fprintf (stdout, "Failure\n");
		break;
	case 0:
		fprintf (stdout, "IDAC Calibration has yet to be run\n");
		break;
	case 1:
		fprintf (stdout, "Success\n");
		break;
	default:
		break;
	}

	return EOK;
}

int
cypress_request_test_data_writen_to_file(int coid)
{
	int reply;
	cypress_msg_u msg;

	msg.cmd = CYPRESS_CTRL_SAVE_CP_CM_DATA_TO_FILE;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), NULL, 0);

	if (EOK != reply) {
		printf ("Writing test data to file failed; reply=%d\n", reply);
		return -1;
	}

	return 0;
}

int
cypress_read_file(int coid, char *filename)
{
	char line [1024];
	FILE *file = fopen (filename, "r");

	if (file != NULL) {
		while (fgets (line, sizeof line, file) != NULL) {
			fputs (line, stdout);
		}

		fclose (file);
	} else {
		fprintf (stderr, "Failed to open the file %s.\n", filename);
		perror (filename);
	}

	return 0;
}

int
cypress_init_baselines (int sensing_mode, int coid)
{
        int reply;
        cypress_msg_u msg;

	if (0 == sensing_mode) {
		fprintf (stderr, "Invalid sensing mode: %d\n", sensing_mode);
		return -1;
	}

        msg.cmd = CYPRESS_CTRL_INIT_BASELINES;
        msg.ctrl.data.sensing_mode = sensing_mode;

        fprintf (stderr, "Init of Baselines requested, sensing mode: %x\n", msg.ctrl.data.sensing_mode);

        DEBUG_LOG("msg.cmd=%d", msg.cmd);

        reply = MsgSend(coid, &msg, sizeof(msg), NULL, 0);

        if (EOK != reply) {
                printf ("Failed to send Init Baselines command; reply=%d\n", reply);
                return -1;
        }

        return 0;
}

int
cypress_request_idac_data(int sensing_mode, int coid)
{
#if 0
	int reply;
	int size = 0;
     uint8_t *idac_data;
	// cypress_msg_u msg;

	/* Note: Currently unsupported!! */

	/* Request the Data Size */
	msg.cmd = CYPRESS_CTRL_IDAC_DATA_SIZE_REQUEST;
	msg.ctrl.data.sensing_mode = sensing_mode;

	fprintf (stderr, "Requesting Customer Data\n");

	fprintf (stderr, "Sending message type %d\n", msg.cmd);
	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof (msg), &size, sizeof (size));
	if (EOK != reply) {
		printf ("Request for Design Data size failed; reply=%d\n", reply);
		return -1;
	}

	/* Get the data */
	idac_data = calloc (size, sizeof(uint8_t));

	msg.cmd = CYPRESS_CTRL_IDAC_DATA_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);
	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof (msg), idac_data, size);
	if (EOK != reply) {
		printf ("Request for Design Data failed; reply=%d\n", reply);
		return -1;
	}

	/* Dump out the contents of the register */
	dump_data (idac_data, size);
	free (idac_data);
#endif

	return EOK;
}


int
cypress_request_customer_data(int coid)
{
	int reply;
	int size = 0;
	uint8_t *design_data, *manufacturer_data;
	cypress_msg_u msg;

	/* Request the Design Data */
	msg.cmd = CYPRESS_CTRL_DESIGN_SIZE_REQUEST;

	fprintf (stderr, "Requesting Customer Data\n");

	fprintf (stderr, "Sending message type %d\n", msg.cmd);
	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof (msg), &size, sizeof (size));
	if (EOK != reply) {
		printf ("Request for Design Data size failed; reply=%d\n", reply);
		return -1;
	}

	design_data =(uint8_t *) calloc (size, sizeof(uint8_t));

	msg.cmd = CYPRESS_CTRL_DESIGN_DATA_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);
	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof (msg), design_data, size);
	if (EOK != reply) {
		printf ("Request for Design Data failed; reply=%d\n", reply);
		return -1;
	}

	/* Dump out the contents of the register */
	dump_data (design_data, size);
	free (design_data);

	/* Request the Manufacturer Data */
	msg.cmd = CYPRESS_CTRL_MANUFACTURER_SIZE_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);
	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof (msg), &size, sizeof (size));
	if (EOK != reply) {
		printf ("Request for Manufacturer Data size failed; reply=%d\n", reply);
		return -1;
	}

	manufacturer_data =(uint8_t *) calloc (size, sizeof(uint8_t));

	msg.cmd = CYPRESS_CTRL_MANUFACTURER_DATA_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);
	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof (msg), manufacturer_data, size);
	if (EOK != reply) {
		printf ("Request for Manufacturer Data failed; reply=%d\n", reply);
		return -1;
	}

	/* Dump out the contents of the register */
	dump_data (manufacturer_data, size);
	free (manufacturer_data);

	return EOK;
}

int
cypress_controller_info(int coid)
{
	int reply;
	cypress_msg_u msg;
	cypress_msg_controller_info_t controller_info;

	msg.cmd = CYPRESS_CTRL_CONTROLLER_INFO_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), &(controller_info), sizeof (controller_info));

	if (EOK != reply) {
		printf ("Request for controller info failed; reply=%d\n", reply);
		return -1;
	}

	fprintf (stdout, "Cypress Controller Info:\n");
	fprintf (stdout, "Product ID: %d\n", controller_info.product_id);
	fprintf (stdout, "Firmware Version: %d.%d\n", controller_info.firmware_ver_major, controller_info.firmware_ver_minor);
	fprintf (stdout, "Silicon ID: %x\n", controller_info.silicon_id);

	return 0;
}

int
cypress_additional_controller_info(int coid)
{
	int reply;
	cypress_msg_u msg;
	cypress_msg_add_controller_info_t controller_info;

	msg.cmd = CYPRESS_CTRL_ADDTIONAL_CONTROLLER_INFO_REQUEST;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), &(controller_info), sizeof (controller_info));

	if (EOK != reply) {
		printf ("Request for controller info failed; reply=%d\n", reply);
		return -1;
	}

	fprintf (stdout, "Additional Cypress Controller Info:\n");
	fprintf (stdout, "Rev Control Number %lu\n", controller_info.revctrl);
	fprintf (stdout, "Version %d.%d\n", controller_info.bl_ver_major, controller_info.bl_ver_minor);
	fprintf (stdout, "Manufacturing ID Size %d\n", controller_info.mfgid_sz);

	fprintf (stdout, "Manufacturing ID Byte:\n\t");
	dump_data (controller_info.mfgid, controller_info.mfgid_sz);

	fprintf (stdout, "CYITO_VER %d\n", controller_info.cyito_ver);
	fprintf (stdout, "TTSP_VER %d.%d\n", controller_info.ttsp_ver_major, controller_info.ttsp_ver_minor);
	fprintf (stdout, "%s Endian\n", controller_info.device_info ? "Little" : "Big");

	return 0;
}

int
cypress_request_noise_data(int coid)
{
	int reply;
	cypress_msg_u msg;
	cypress_msg_retrieve_noise_data_t noise_data;

	msg.cmd = CYPRESS_CTRL_NOISE_LEVEL_DATA_REQUEST;

	fprintf (stdout, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), &(noise_data), sizeof (noise_data));

	if (EOK != reply) {
		fprintf (stderr, "Request for controller info failed; reply=%d\n", reply);
		return -1;
	}

	fprintf (stdout, "Noise Level: %d\n", noise_data.noise_level);

	if (noise_data.noise_level_velocity) {
		fprintf (stdout, "Noise level velocity %d\n", noise_data.noise_level_velocity);
	} else {
		fprintf (stdout, "Noise level velocity not supported on this hardware\n");
	}

	return 0;
}


int
cypress_reset_controller(int coid)
{
	int reply;
	cypress_msg_u msg;

	msg.cmd = CYPRESS_CTRL_RESET;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), NULL, 0);

	if (EOK != reply) {
		printf ("Failed to issue a reset; reply=%d\n", reply);
		return -1;
	}

	return 0;
}

int
cypress_last_xy(int coid, unsigned *x, unsigned *y)
{
	int reply;
	cypress_msg_u msg;
	point_t last_coords;

	msg.cmd = CYPRESS_CTRL_LASTXY_DATA;

	fprintf (stderr, "Sending message type %d\n", msg.cmd);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), &last_coords, sizeof (last_coords));

	if (EOK != reply) {
		printf ("Failed to get last coordinates, possibly none available yet; reply=%d\n", reply);
		return -1;
	}

	fprintf (stdout, "Last sent coordinates were X%d Y%d\n", last_coords.x, last_coords.y);

	return 0;
}

int
cypress_calibrate_idacs (int sensing_mode, int coid)
{
	int reply;
	cypress_msg_u msg;

	msg.cmd = CYPRESS_CTRL_IDAC_CALIBRATION_REQUIRED;
	msg.ctrl.data.sensing_mode = sensing_mode;

	fprintf (stderr, "Calibration of IDACs requested, sensing mode: %x\n", msg.ctrl.data.sensing_mode);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), NULL, 0);

	if (EOK != reply) {
		printf ("Failed to send upgrade firmware command; reply=%d\n", reply);
		return -1;
	}

	return 0;
}

int
cypress_calibrate_idacs_now (int sensing_mode, int coid)
{
	int reply;
	cypress_msg_u msg;

	msg.cmd = CYPRESS_CTRL_IDAC_CALIBRATION_REQUEST;
	msg.ctrl.data.sensing_mode = sensing_mode;

	fprintf (stderr, "Calibration of IDACs requested, sensing mode: %x\n", msg.ctrl.data.sensing_mode);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), NULL, 0);

	if (EOK != reply) {
		printf ("IDAC calibration request failed. reply=%d\n", reply);
		return -1;
	}

	return 0;
}

int
cypress_firmware_upgrade(int sensing_mode, int coid, char *file)
{
	int reply;
	cypress_msg_u msg;

	/* Sensing mode has been set, so we will calibrate IDAC's */
	if (sensing_mode) {
		if (-1 == cypress_calibrate_idacs (sensing_mode, coid)) {
			fprintf (stderr, "Failed to set sensing modes on touch driver, aborting firmware update\n");
		}
	}

	msg.cmd = CYPRESS_CTRL_FIRMWARE_UPGRADE;
	strncpy (msg.ctrl.data.buf, file, sizeof(msg.ctrl.data.buf)-1);

	fprintf (stderr, "Sending message type %d file %s\n", msg.cmd, msg.ctrl.data.buf);

	DEBUG_LOG("msg.cmd=%d", msg.cmd);

	reply = MsgSend(coid, &msg, sizeof(msg), NULL, 0);

	if (EOK == reply) {
		printf ("Firmware upgrade successful\n");
	} else {
		printf ("Firmware upgrade failed: reply=%d\n", reply);
		return -1;
	}

	return 0;
}



int main(int argc, char* argv[])
{
	int c, coid = -1;
	unsigned last_x, last_y;
	int firmware_upgrade = 0;
	int idac_calibration = 0;
	int idac_data_request = 0;
	int sensing_mode = 0;
	int init_baselines = 0;
	char *file = NULL;
	// int choice = 0;

	coid = name_open(ATTACH_POINT, 0);
	if (coid == -1) {
		DEBUG_LOG("can't open '%s'", ATTACH_POINT);
		return 1;
	}

	while ((c = getopt (argc, argv, "f:c:lrm:iaIDnbR:Ws")) != -1) {
		switch (c) {
		case 'f':
			firmware_upgrade = 1;
			file = strdup (optarg);
			break;
		case 'l':
			cypress_last_xy (coid, &last_x, &last_y);
			break;
		case 'r':
			cypress_reset_controller (coid);
			break;
		case 'm':
			sensing_mode = strtoul (optarg, NULL, 10);
			break;
		case 'i':
			cypress_controller_info(coid);
			cypress_request_customer_data(coid);
			break;
		case 'a':
			cypress_additional_controller_info(coid);
			break;
		case 'I':
			idac_calibration = 1;
			break;
		case 'D':
			idac_data_request = 1;
			break;
		case 'n':
			cypress_request_noise_data(coid);
			break;
		case 'b':
			init_baselines = 1;
			break;
		case 'R':
			file = strdup (optarg);
			cypress_read_file(coid, file);
			break;
		case 'W':
			cypress_request_test_data_writen_to_file(coid);
			break;
		case 's':
			cypress_request_controller_status (coid);
			break;
		default:
			fprintf (stderr, "invalid or no options specified\nvalid options are -s (for stopping) and -a (for awaking)\n");
			return 1;
		}
	}

	if (firmware_upgrade) {
		if (file != NULL)
			cypress_firmware_upgrade (sensing_mode, coid, file);
	}

	if (idac_calibration) {
		cypress_calibrate_idacs_now (sensing_mode, coid);
	}

	if (idac_data_request) {
		cypress_request_idac_data (sensing_mode, coid);
	}

	if (init_baselines) {
		cypress_init_baselines (sensing_mode, coid);
	}

	name_close(coid);

	return 0;
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/cypress/cypress_ctrl/main.c $ $Rev: 886397 $")
#endif
